#!/usr/bin/env python
'''
Copyright (C) 2006 Jean-Francois Barraud, barraud@math.univ-lille1.fr

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
barraud@math.univ-lille1.fr

Quick description:
The extensions use all selected path, ignoring all other selected objects.
It tries to regularize hand drawn path by the following :
 - finds sequences of aligned points and replace them by a simple segment.
 - changes the segments angles to the closest remarkable angle (pi/2, pi/3, pi/6, etc...)
 - eqalizes all segments lenghts which are close to each other
 - replaces 4 segments paths by a rectangle if this makes sens (giving the correct rotation to the rectangle). 

'''

import inkex, cubicsuperpath, bezmisc
import pathmodifier,simpletransform, simplepath
import copy, math, re, random
import gettext
_ = gettext.gettext




import numpy
numpy.set_printoptions(precision=1)
def toArray(parsedList):
    interpretCommand = {
        'C' : lambda x : x[-2:],
        'L' : lambda x : x[0:2],
        'M' : lambda x : x[0:2],
        }
    a=numpy.array([ interpretCommand[cmd](points)  for (cmd, points) in parsedList  ])
    return a

rotMat = numpy.matrix( [[1,-1],[1,1]] )/numpy.sqrt(2)
unrotMat = numpy.matrix( [[1,1],[-1,1]] )/numpy.sqrt(2)

def setupKnownAngles():
    pi = numpy.pi
    l = [ i*pi/8 for i in range(0, 9)] +[ i*pi/6 for i in [1,2,4,5,] ]
    knownAngle = numpy.array( l )
    return numpy.concatenate( [-knownAngle[:0:-1], knownAngle ])
knownAngle = setupKnownAngles()

from numpy import sqrt

def D2(p1, p2):
    return ((p1-p2)**2).sum()


def void(*l):
    pass
def debug_on(*l):
    print ' '.join(str(i) for i in l)
debug = void

def set_debug_on():
    global debug
    debug = debug_on
    
def computeBox(a):
    #print a    
    xmin , ymin = a[:,0].min(), a[:,1].min()
    xmax , ymax = a[:,0].max(), a[:,1].max()

    return xmin, ymin, xmax-xmin, ymax-ymin

def dirAndLength(p1,p2):
    l = max(sqrt( D2(p1, p2) ),1e-4)
    uv = (p1-p2)/l
    return l,uv

def length(p1,p2):
    return sqrt( D2(p1,p2) )


class Path(object):
    next = None
    prev = None
    startIndexSource = 0
    segmentMergable = False
    sourcepoints = None
    
    def __init__(self, points):
        self.points = points
        self.init()

    def init(self):
        self.effectiveNPoints = len(self.points)
        if self.effectiveNPoints>1:
            self.length , self.univ = dirAndLength(self.points[0], self.points[-1])
        else:
            self.length , self.univ = 0, numpy.array([0,0])
    def isSegment(self):
        return False
    def quality(self):
        return 1000

    def dump(self):
        n = len(self.points)
        if n>0:
            return 'path at '+str(self.points[0])+ ' to '+ str(self.points[-1])+'    npoints=%d / %d (eff)'%(n,self.effectiveNPoints)
        else:
            return 'path Void !'
        

    def removeLastPoints(self,n):
        self.points = self.points[:-n]
        self.init()
    def removeFirstPoints(self,n):
        self.points = self.points[n:]
        self.startIndexSource += n
        self.init()

    def costheta(self,seg):
        return self.unitv.dot(seg.unitv)

    def translate(self, tr):
        """Translate this path by tr"""
        #self.c = -self.c -self.a*tr[0] -self.b*tr[0]
        self.points = self.points + tr
        #self.pointN = self.pointN+tr
        #self.point1 = self.point1+tr


    
class Segment(Path):
    """ a segment. Defined by its line equation ax+by+c=0 and the points from orignal paths
    it is ensured that a**2+b**2 = 1
    """
    QUALITYCUT = 0.9
    
    segmentMergable = True
    newAngle = None
    newDistance = None

    _nseg = 0
    
    def __init__(self, a,b,c, points, doinit=True):

        self.a = a
        self.b = b
        self.c = c

        self.sid = Segment._nseg
        Segment._nseg +=1
        
        self.points = points
        d = numpy.sqrt(a**2+b**2)
        if d != 1. :
            self.a /= d
            self.b /= d
            self.c /= d

        if doinit :
            self.init()

    def init(self):

        a,b,c = self.a, self.b, self.c
        x,y = self.points[0]
        self.point1 = numpy.array( [ b*(x*b-y*a) - c*a, a*(y*a-x*b) - c*b ] )
        x,y = self.points[-1]
        self.pointN = numpy.array( [ b*(x*b-y*a) - c*a, a*(y*a-x*b) - c*b ] )
        uv = self.computeDirLength()
        self.distancesToLine =  self.computeDistancesToLine(self.points)
        self.normalv = numpy.array( [ a, b ])

        self.angle = numpy.arccos( uv[0] )*numpy.sign(uv[1] )

    def computeDirLength(self):
        self.length , uv = dirAndLength(self.pointN, self.point1)
        self.unitv = uv
        return uv

    def recomputeEndPoints(self):

        a,b,c = self.a, self.b, self.c
        x,y = self.points[0]
        self.point1 = numpy.array( [ b*(x*b-y*a) - c*a, a*(y*a-x*b) - c*b ] )
        x,y = self.points[-1]
        self.length = numpy.sqrt( D2(self.pointN, self.point1) )

    def projectPoint(self,p):
        a,b,c = self.a, self.b, self.c
        x,y = p
        return numpy.array( [ b*(x*b-y*a) - c*a, a*(y*a-x*b) - c*b ] )        
        
    def pointAtX(self, x):
        return numpy.array( ( x , x*self.slope+self.offset) )

    def intersect(self, seg):
        nu, nv = self.normalv, seg.normalv
        #debug( ' XXX intersect  ',self.dumpShort() )
        #debug( '     with ',seg.dumpShort() )
        u = numpy.array([[-self.c],[-seg.c]])
        doRotation = min(nu.min(),nv.min()) <1e-4
        if doRotation:
            # rotate to avoid numerical issues
            nu = numpy.array(rotMat.dot(nu))[0]
            nv = numpy.array(rotMat.dot(nv))[0]
        m = numpy.matrix( (nu, nv) )        

        i =  (m**-1*u) 
        if doRotation:
            i = unrotMat*i
            
        i=numpy.array( i).swapaxes(0,1)[0]
        #print m, u , ' ---> ',(m**-1*u).swapaxes(0,1)[0][0], i.swapaxes(1,0)[0:1]
        return i
        #return (m**-1*u).swapaxes(0,1)[0]
        #return (m**-1*u)[0]

    def setIntersectWithNext(self, next=None):
        if next is None:
            next = self.next
        if next and next.isSegment():
            inter = self.intersect(next)
            debug(' Intersect at ',inter, ' from ',self.dump())
            next.point1 = inter
            self.pointN = inter
            self.computeDirLength()
            next.computeDirLength()
            #next.length = numpy.sqrt(D2(inter,next.pointN))
            #self.length = numpy.sqrt(D2(inter,self.point1))
            
    def computeDistancesToLine(self, points):
        return abs(self.a*points[:,0]+self.b*points[:,1]+self.c)


    def distanceTo(self,point):
        return abs(self.normalv.dot(points)+offset)
        

    def isSegment(self):
        return True

    def inverse(self):
        def inv(v):
            v[0], v[1] = v[1] , v[0]
        for v in [self.point1 , self.pointN , self.unitv, self.normalv]:
            inv(v)

        self.points = numpy.roll(self.points,1,axis=1)
        self.a, self.b = self.b, self.a
        self.angle = numpy.arccos( self.unitv[0] )*numpy.sign(self.unitv[1] )
        return

    def dumpShort(self):
        return 'seg  '+str(self.startIndexSource)+'  '+str(self.point1 )+'to '+str(self.pointN)+ ' npoints=%d | angle,offset=(%.2f,%.2f )'%(len(self.points),self.angle, self.c)+'  ',self.normalv

    def dump(self):
        v = self.variance()
        n = len(self.points)
        return 'seg  '+str(self.point1 )+'to '+str(self.pointN)+ '  v/l=%.2f / %.2f = %.2f  r*sqrt(n)=%.2f  npoints=%d | angle,offset=(%.2f,%.2f )'%(v, self.length, v/self.length,v/self.length*numpy.sqrt(n) ,n  , self.angle, self.c)
        
    def variance(self):
        d = self.distancesToLine
        return numpy.sqrt( (d**2).sum()/len(d) )

    def quality(self):
        n = len(self.points)
        return min(self.variance()/self.length*numpy.sqrt(n) , 1000)

    def formatedSegment(self, firstP=False):
        if firstP:            
            segment = [ ['M',[self.point1[0],self.point1[1] ] ],
                        ['L',[self.pointN[0],self.pointN[1] ] ]
                        ]
        else:
            segment = [ ['L',[self.pointN[0],self.pointN[1] ] ] ] 
        return segment
        
    def replaceInList(self, startPos, fullList):
        code0 = fullList[startPos][0]
        segment = [ [code0,[self.point1[0],self.point1[1] ] ],
                     ['L',[self.pointN[0],self.pointN[1] ] ]
                    ]
        l = fullList[:startPos]+segment+fullList[startPos+len(self.points):]
        return l




    def mergedWithNext(self):
        """ Returns the combination of self and self.next.
        sourcepoints has to be set
        """
        spoints = self.sourcepoints[self.startIndexSource:self.startIndexSource+len(self.points)+len(self.next.points)]

        newSeg = fitSingleSegment(spoints)
        newSeg.startIndexSource = self.startIndexSource
        newSeg.prev = self.prev
        if self.prev : newSeg.prev.next = newSeg
        newSeg.next = self.next.next
        if newSeg.next: newSeg.next.prev = newSeg
        return newSeg

    def extendWith(self, points, backward=False):
        """deprecated ? """
        d2sum = (self.distancesToLine**2).sum()
        l = self.length
        np = len(self.points)
        uv = self.unitv
        if backward:
            refpoint = self.pointN
            points = points[::-1] # create a reverse view
            uv = -uv
        else:
            refpoint = self.point1
        lastInd = 0
        q = self.quality()
        ponline = refpoint
        QCut = Segment.QUALITYCUT
        for p in points:
            d2=self.distanceTo(p)**2
            #newV= /(l+1)
            vtoLast = p - ponline
            vtoLast = vtoLast
            costheta = uv.dot(vtoLast) # this has the sign of the costheta
            
            #ponline = (p[0], p[0]*self.slope+self.offset)
            ponline = seg.projectPoint(p)
            newl = numpy.sqrt( D2(refpoint, ponline) )
            newq= numpy.sqrt( (d2sum + d2))/newl
            #debug('  extednd to ',p ,' d2sum ',d2sum, '  d2',d2 , '  newq ',newq, ' / ',q , ' newl=',newl ,' /',l)            
            if newq < 2*q and newq< QCut and costheta>0:
                d2sum += d2
                l = newl
                np+=1
                lastInd +=1
                continue
            else:
                break

            
        extending = points[:lastInd]
        if backward:
            extending = extending[::-1]
        return extending
    
    def refit(self):
        xmin,ymin,w,h = computeBox(self.points)
        inverse = w<h
        if inverse:
            self.points = numpy.roll(self.points,1,axis=1) 

        s , o = regLin(self.points, returnOnlyPars=True)
        self.a=s
        self.b=-1
        self.c = o
        self.init()
        if inverse:
            self.inverse()

    def barycenter(self):
        return 0.5*(self.point1+self.pointN)

    def translate(self, tr):
        """Translate this segment by tr"""
        self.c = -self.c -self.a*tr[0] -self.b*tr[0]
        self.pointN = self.pointN+tr
        self.point1 = self.point1+tr
        
    def adjustToNewAngle(self):        
        self.a,self.b,self.c = parametersFromPointAngle( self.point1, self.newAngle)
        self.angle = self.newAngle
        self.normalv = numpy.array( [ self.a, self.b ])
        self.unitv = numpy.array( [ self.b, -self.a ])
        if abs(self.angle) > numpy.pi/2 :
            if self.b > 0: self.unitv *= -1
        elif self.b<0 : self.unitv  *= -1
        #print 'adjusted angle ', self.normalv

    def adjustToNewDistance(self):
        #print ' ___ adjustToNewDistance ', self.point1, self.pointN, self.newDistance, self.unitv
        self.pointN = self.newDistance* self.unitv + self.point1
        #print ' ___ adjustToNewDistance ', self.point1, self.pointN, self.newDistance, self.unitv
        self.length = self.newDistance
        #pass

    def tempLength(self):
        if self.newDistance : return self.newDistance
        else : return self.length

class Rectangle(object):
    def __init__(self, center, size, angle):
        self.center = center
        self.size = size
        self.angle = angle
        cosa = numpy.cos(angle)
        sina = numpy.sin(angle)
        if angle != 0:            
            self.rotMat = numpy.matrix( [ [ cosa, sina], [-sina, cosa] ] )
            rcenter = numpy.array(self.rotMat.dot(self.center))[0]
            debug( ' !!! rotated rect ', self.center, ' -> ', rcenter)
            self.center = rcenter
            self.rotMatstr = 'matrix(%1.7f,%1.7f,%1.7f,%1.7f,0,0)'%(cosa, sina, -sina, cosa)
        else :
            self.rotMatstr = None
            
    def fill(self,ele):
        w, h = self.size
        ele.set('width',str(w))
        ele.set('height',str(h))
        ele.set('x',str(self.center[0]-w/2))
        ele.set('y',str(self.center[1]-h/2))
        if self.rotMatstr:
            ele.set('transform', self.rotMatstr)
        ele.set('style', 'fill:none;stroke:#FF0000;stroke-opacity:1;stroke-width:3;')


def fitSingleSegment(a):
    xmin,ymin,w,h = computeBox(a)
    inverse = w<h
    if inverse:
        a = numpy.roll(a,1,axis=1)

    seg = regLin(a)
    if inverse:
        seg.inverse()
        #a = numpy.roll(a,1,axis=0)
    return seg
        
def regLin(a , returnOnlyPars=False):
    """perform a linear regression on 2dim array a. Creates a segment object in return """
    sumX = a[:,0].sum()
    sumY = a[:,1].sum()
    sumXY = (a[:,1]*a[:,0]).sum()
    a2 = a*a
    sumX2 = a2[:,0].sum()
    sumY2 = a2[:,1].sum()
    N = a.shape[0]

    pa = (N*sumXY - sumX*sumY)/ ( N*sumX2 - sumX*sumX)
    pb = (sumY - pa*sumX) /N
    if returnOnlyPars:
        return pa, pb
    return Segment(pa, -1, pb, a)

def adjustAllAngles(paths):
    for p in paths:
        if p.isSegment() and p.newAngle is not None:
            p.adjustToNewAngle()
            if p.next is None or not p.next.isSegment():
                # move the last point (no intersect with next)

                pN = p.projectPoint(p.pointN)
                dirN = pN - p.point1                
                lN = length(pN, p.point1)
                p.pointN = dirN/lN*p.length + p.point1
                #print ' ... adjusting last seg angle ',p.dump() , ' normalv=', p.normalv, 'unitv ', p.unitv
            else:
                p.setIntersectWithNext()
    # next translate to fit end points
    tr = numpy.zeros(2)
    for p in paths[1:]:
        if p.isSegment() and p.prev.isSegment():
            tr = p.prev.pointN - p.point1
        #debug( 'translating of ', tr, p.dump())
        p.translate(tr)

def adjustAllDistances(paths):
    for p in paths:
        if p.isSegment() and  p.newDistance is not None:                
            p.adjustToNewDistance()
            #print ' ... adjusted distance ',p.dump() , ' normalv=', p.normalv, 'unitv ', p.unitv
    # next translate to fit end points
    tr = numpy.zeros(2)
    for p in paths[1:]:
        if p.isSegment() and p.prev.isSegment():
            tr = p.prev.pointN - p.point1
        #debug( 'translating of ', tr, p.dump())
        p.translate(tr)

def adjustPath(paths):
    # first adjust angles, distances
    for p in paths:
        if p.isSegment():
            if p.newAngle is not None:
                p.adjustToNewAngle()
                if p.next is None or not p.next.isSegment():
                    # move the last point (no intersect with next)

                    pN = p.projectPoint(p.pointN)
                    dirN = pN - p.point1
                    lN = length(pN, p.point1)
                    p.pointN = dirN/lN*p.length + p.point1
                    #print ' ... adjusting last seg angle ',p.dump() , ' normalv=', p.normalv, 'unitv ', p.unitv
                else:
                    p.setIntersectWithNext()
            if p.newDistance is not None:                
                p.adjustToNewDistance()
                #print ' ... adjusted distance ',p.dump() , ' normalv=', p.normalv, 'unitv ', p.unitv
    # next translate to fit end points
    tr = numpy.zeros(2)
    for p in paths[1:]:
        if p.isSegment() and p.prev.isSegment():
            tr = p.prev.pointN - p.point1
        #debug( 'translating of ', tr, p.dump())
        p.translate(tr)
    
            

def parametersFromPointAngle(point, angle):
    unitv = numpy.array([ numpy.cos(angle), numpy.sin(angle) ])
    ortangle = angle+numpy.pi/2
    normal = numpy.array([ numpy.cos(ortangle), numpy.sin(ortangle) ])
    genOffset = -normal.dot(point)
    a, b = normal
    return a, b , genOffset
    
def segmentFromPointsAngle(points, angle):
    barycenter = points.sum(axis=0)/len(points)
    a, b , offset = parametersFromPointAngle(barycenter, angle)
    return Segment(a,b,offset,points)
        

#class Myext(pathmodifier.Diffeo):
class SegmentRec(inkex.Effect):
    def __init__(self):
        inkex.Effect.__init__(self)
        self.OptionParser.add_option("--title")
        self.OptionParser.add_option("-n", "--noffset",
                        action="store", type="float", 
                        dest="noffset", default=0.0, help="normal offset")
        self.OptionParser.add_option("-t", "--toffset",
                        action="store", type="float", 
                        dest="toffset", default=0.0, help="tangential offset")
        self.OptionParser.add_option("-k", "--kind",
                        action="store", type="string", 
                        dest="kind", default=True,
                        help="choose between wave or snake effect")
        self.OptionParser.add_option("-c", "--copymode",
                                     action="store", type="string", 
                        dest="copymode", default=True,
                        help="repeat the path to fit deformer's length")
        self.OptionParser.add_option("-p", "--space",
                        action="store", type="float", 
                        dest="space", default=0.0)
        self.OptionParser.add_option("-v", "--vertical",
                        action="store", type="inkbool", 
                        dest="vertical", default=False,
                        help="reference path is vertical")
        self.OptionParser.add_option("-d", "--duplicate",
                        action="store", type="inkbool", 
                        dest="duplicate", default=False,
                        help="duplicate pattern before deformation")
        self.defPathName = ['path2985']



    def effect(self):

        rej='{http://sodipodi.sourceforge.net/DTD/sodipodi-0.dtd}type'
        paths = []
        for id, node in self.selected.iteritems():
            if node.tag == '{http://www.w3.org/2000/svg}path' and rej not in node.keys():                
                paths.append(node)
        if paths == []:
            paths=[self.getElementById(n) for n in self.defPathName]
        else:
            p = paths[0]

        self.extractShapes(paths)
        


    def removeSmallEdge(self, paths):
        """Remove small Path objects which stand between 2 Segments (or at the ends of the sequence).
        Small means the bbox of the path is less then 5% of the mean of the 2 segments."""
        if len(paths)<2:
            return
        def getdiag(points):
            xmin,ymin,w,h = computeBox(p.points)
            return sqrt(w**2+h**2)
        
        for p in paths:
            if p.isSegment(): continue
            if len(p.points)==0 : continue
            # select only path between 2 segments
            next, prev = p.next, p.prev
            if next is None: next = prev
            if prev is None: prev = next
            if not next.isSegment() or not prev.isSegment() : continue
            diag = getdiag(p.points)

            debug(' removing edge  diag = ', diag, '  l=',next.length+prev.length)
            if diag > (next.length+prev.length)*0.1: continue
            debug('   --> remove !')


            p.effectiveNPoints = 0
            if next != prev:
                prev.setIntersectWithNext(next)


    def distancesRatios(self, points):
        """ Finds segment-like portions in a sequence of points.
        input : points is a (N,2) shaped array
        output : a list of Segment or Path objects

        The idea used here is for aligned points D(p_i,p_n) = Sum(D_i,D_i-1) = S
        So start from point 1, aggregate all following points as long as S is close to D, when this is not true or D(p_1,P_n)<D(p_1,p_n-1)
        start a new segment.
        All points in segments of length less than 4 are simply put into generic Path.
        """
        #points = self.origPoints
        Np = len(points)
        windowS= min( max( Np/10, 2) , 10 )

        presegs = []
        i=0
        def pdistance(i,j):
            return numpy.sqrt( ((points[i]-points[j])**2).sum() )

        # aggregates points into candidate segments (presegs)
        while i < Np-1:
            # start a new preseg with point i
            next = i
            #sumD = dmat[i,next]
            sumD =  0
            lastD = sumD
            curD = sumD
            r = 1
            # loop on next points as long as they're compatible with a segment.
            while r < 1.15 and curD >= lastD and next< Np-1:
                next += 1
                #curD = dmat[next-1,neaxt]
                lastD =curD
                curD = max(pdistance(i,next) , 1e-12)
                sumD += pdistance(next-1,next)
                r = sumD/curD
            # mark the start and end position of the pre-segment
            next -=1            
            presegs.append( (i,next) )
            #print (i,next) ,'  r=',r
            i = next +1
        presegs[ -1 ] = ( presegs[-1][0],presegs[-1][1]+1 ) #append the last point

        #print 'presegs ', presegs
        #  convert to Segment:
        minNpointInSeg = min(4,max(Np/20,2))
        segs = []
        for (b,e) in presegs:
            if e+1-b<4: # less than 4 points, not a Segment
                seg = Path(points[b:e+1])
            else: # comute the segment (fit a line equation)
                seg = fitSingleSegment(points[b:e+1])
                debug('  seg at ',(b,e),seg.dump())
                if seg.quality() > Segment.QUALITYCUT : # fit failed. Not a segment
                    seg = Path(points[b:e+1])
            seg.startIndexSource = b
            seg.sourcepoints = points
            segs.append( seg )
        # assign next and previous :
        for p,pnext in zip(segs[:-1] , segs[1:]):
            p.next = pnext
            pnext.prev = p
        if len(segs)>1 :segs[-1].prev = segs[-2]

            

        # merge consecutive Path objects 
        mergedpaths = []
        nseg = len(segs)
        def mergeConsecutivePath(p, pathStart=0):
            nexts = 0
            nextp = p.next
            if  p.isSegment() :
                yield p
                if p.next is None:
                    return
                nexts = p.next.startIndexSource
            elif p.next is None or p.next.isSegment() :
                newp = Path( points[pathStart:p.startIndexSource+p.effectiveNPoints] )                
                newp.startIndexSource = pathStart
                yield newp
                if p.next is None:
                    return
                nexts = p.next.startIndexSource
            else:
                nexts = pathStart
            #print ' merging cont  ... ',nextp.dump(), nexts
            for op in mergeConsecutivePath(nextp, nexts):
                yield op

        segs =[ p for p in mergeConsecutivePath(segs[0],0) ]    
        
        # merge consecutive segments if they have similar angles
        # compute delta angles
        for p in segs:
            if p.isSegment() and p.next and p.next.isSegment():                
                p.deltaAngle = abs(p.angle - p.next.angle)
            else:
                p.deltaAngle = 10
        pmin = min( segs, key = lambda x:x.deltaAngle)
        while pmin.deltaAngle < 0.13962: # degree
            spoints = points[pmin.startIndexSource:pmin.next.startIndexSource+len(pmin.next.points)]
            newSeg  = pmin.mergedWithNext(  ) 
            if newSeg.next and newSeg.next.isSegment():
                newSeg.deltaAngle = abs(newSeg.angle - newSeg.next.angle)
            else:
                newSeg.deltaAngle = 10
            segs.remove(pmin)
            segs.remove(pmin.next)
            segs.append(newSeg)
            pmin = min( segs, key = lambda x:x.deltaAngle)
        
        return segs
            
            
    
    def prepareParrallelize(self,paths):
        """Group Segment by their angles (segments are grouped together if their deltAangle is within 0.02 rad)
        The angles of segments in a group are then set to the mean angle of the group"""
        segs = [ seg for seg in paths if seg.isSegment()]
        closeAngles = {}
        for (i, seg1) in enumerate(segs):
                for seg2 in segs[i+1:]:
                    if abs(seg1.angle - seg2.angle)< 0.02:
                        l1=closeAngles.get(seg1,set())
                        l1.add(seg1)
                        l1.add(seg2)
                        closeAngles[seg2] = l1
                        debug('parralelize ',seg1.dump() , seg2.dump())
        for segs in closeAngles.values():
            if segs == set():
                return
            n = len(segs)
            meanAngle = sum( s.angle for s in segs) / n
            for s in segs:
                s.newAngle = meanAngle

    def prepareDistanceEqualization(self,paths, relDelta=0.1):
        """ Segmemts in paths are grouped according to their length according to this procedure :
          - for each lenght L, find all other lenghts within L/10. of L.
          - Find the larger of such subgroup.
          - repeat the procedure on remaining lenghts until none is left.
        Each length in a group is set to the mean length of the group
        """
        segs = [ seg for seg in paths if seg.isSegment()]
        lengths= sorted(  (x.tempLength() ,i) for i,x in enumerate(segs)  )
        debug( '___  lenghts ',lengths)

        def findgroups(ls, startPos=0 ):
            maxgroupdelta = 0
            maxgrouppos = ()
            #print '  ccc find groups in ', ls
            for i,d in enumerate(ls):
                thisD = d[0]
                delta = thisD*relDelta
                first, last = i-1,i+1
                while first>-1:
                    if thisD-ls[first][0] > delta:
                        break
                    first-=1
                first +=1
                while last<len(ls):
                    if ls[last][0]-thisD > delta:
                        break
                    last+=1
                last -= 1
                #print i,'     --> ',last , first, delta
                if last-first > maxgroupdelta:
                    maxgroupdelta = last-first
                    maxgrouppos = (first,last)
            #print ' ====> maxgroupdelta ',maxgroupdelta, ' // ',maxgrouppos
            l = []
            if maxgroupdelta >0 :
                l += [ (maxgrouppos[0]+startPos, maxgrouppos[1]+startPos) ]
                #groups.append(maxgrouppos)
                l+= findgroups( ls[:maxgrouppos[0]], startPos)
                l+=findgroups( ls[maxgrouppos[1]+1:], startPos+maxgrouppos[1]+1)
            return l

        sameDistGroups = findgroups(lengths)
        #print '  ___  dist group ', sameDistGroups
        for f,l in sameDistGroups:
            dmean = sum( d for (d,i) in lengths[f:l+1] )/ (l-f+1)
            for d,i in lengths[f:l+1]:
                segs[i].newDistance = dmean
                debug( i,' set newDistance ',dmean, segs[i].length, segs[i].dumpShort())
        

    def adjustToKnownAngle(self, paths):
        for seg in paths:
            if seg.isSegment():
                i = (abs(knownAngle - seg.angle)).argmin()
                debug( '  Known angle ', seg.angle,'  -> ', knownAngle[i]) 
                seg.newAngle = knownAngle[i]


    def simplifyPath(self, parsedpath):
        a = toArray(parsedpath)
        self.currentPoints = a

        def resetPrevNext(segs):
            for i, seg in enumerate(self.segs[:-1]):
                s = self.segs[i+1]
                seg.next = s
                s.prev = seg           

        def distancesRatios():
            self.segs = self.distancesRatios(self.currentPoints)
            self.segs.sort( key = lambda x : x.startIndexSource  )
            resetPrevNext(self.segs)
            self.removeSmallEdge(self.segs)

            return self.segs

        def simpleSeg():
            seg = fitSingleSegment(a)
            seg.startIndexSource = 0
            return [ seg]

        return distancesRatios()


    def isRectangle(self, path):
        path = [p for p in path if p.isSegment() or p.effectiveNPoints >0]
        if len(path) != 4:
            debug( 'rectangle Failed at lenght ', len(path))
            return None
        a,b,c,d = path
        pi , twopi = numpy.pi,2*numpy.pi
        def notCloseAngle( a1, a2) :
            da = abs(a1-a2)
            return min( da , abs(da-pi), abs(da-twopi) ) > 0.001

        if notCloseAngle(a.angle,c.angle) or notCloseAngle(b.angle , d.angle):
            debug( 'rectangle Failed at angles', a.angle, c.angle , b.angle , d.angle)
            return None
        notsimilarL = lambda d1,d2: abs(d1-d2)>0.20*min(d1,d2)
            
        if notsimilarL(a.length, c.length) or notsimilarL(b.length, d.length):
            debug( 'rectangle Failed at distances ', a.length, b.length, c.length, d.length)
            return None

        angles = numpy.array( [p.angle   for p in path] )
        minAngleInd = numpy.argmin( numpy.minimum( abs(angles), abs( abs(angles)-pi), abs( abs(angles)-twopi) ) )
        rotAngle = angles[minAngleInd]
        width = (path[minAngleInd].length + path[(minAngleInd+2)%4].length)*0.5
        height = (path[(minAngleInd+1)%4].length + path[(minAngleInd+3)%4].length)*0.5
        barycenter = numpy.sum( p.barycenter() for p in path)/4.
        r = Rectangle( barycenter, (width, height), rotAngle)
        
        debug( ' found a rectangle !! ', a.length, b.length, c.length, d.length )
        return r

    def extractShapes( self, nodes ):
        paths = []
        for n in nodes :
            parsedList = simplepath.parsePath(n.get('d'))
                      
            simplePaths = self.simplifyPath( parsedList) #distancesRatios()

            paths.append( (parsedList, simplePaths, n) )

        allSegs = [ p  for (pl,sp,n) in paths for p in sp if p.isSegment() ]

        self.prepareParrallelize(allSegs)
        self.adjustToKnownAngle(allSegs)
        for parsedL, path , node in paths:
            # first pass : independently per path
            adjustAllAngles(path)
            self.prepareDistanceEqualization(path, 0.12)
            adjustAllDistances(path)            
        # then 2nd global pass, with tighter criteria
        self.prepareDistanceEqualization(allSegs, 0.05)        
        for parsedL, path , node in paths:
            adjustAllDistances(path)            

        finalpaths = []
        for parsedL, path , node in paths:
            r = self.isRectangle(path)
            if r :
                self.addRectangle( r , node)
            else:
                newList = self.reformatList( parsedL, path)
                self.addPath( newList , node)
                self.lastList = newList
            self.lastPath = path

        


    def reformatList(self, parsedList, paths):
        """ Returns a list in the format of simplepath.parsePath from the original list and the list of extracted Segment and Path objects"""
        newList = []
        first = True
        for  seg in paths:
            if seg.isSegment():
                fseg = seg.formatedSegment(first)
                newList +=fseg
            else:
                if first:
                    pos = seg.startIndexSource
                else:
                    pos = seg.startIndexSource+1
                newList += parsedList[pos:pos+seg.effectiveNPoints]
            first = False
        return newList
    


    def addPath(self,newList, refpath):
        ele = inkex.etree.Element('{http://www.w3.org/2000/svg}path')
        ele.set('d', simplepath.formatPath(newList))
        ele.set('style', 'fill:none;stroke:#FF0000;stroke-opacity:1;stroke-width:3;')
        refpath.xpath('..')[0].append(ele)

    def addRectangle(self, r, refpath):
        ele = inkex.etree.Element('{http://www.w3.org/2000/svg}rect')
        r.fill(ele)
        refpath.xpath('..')[0].append(ele)
    # debugging functions 

    def dumpSegs(self):
        print '-- original points = ', len(self.currentPoints), 'from ', self.currentPoints[0] , ' to  ',self.currentPoints[-1]
        print '  -- divided segs '
        cnt = 0
        for s in self.segs:
            cnt+=len(s.points)
            print s.dump()
        print '   --- > total npoints=',cnt
        print '  -- merged segs '
        cnt = 0
        for s in self.lastPath:
            cnt+=len(s.points)
            print s.dump()
        print '   --- > total npoints=',cnt

    def dumpOldList(self):
        for p in self.oldList:
            print p

        
if __name__ == '__main__':
    e = SegmentRec()
    e.affect()

                    
# vim: expandtab shiftwidth=4 tabstop=8 softtabstop=4 encoding=utf-8 textwidth=99
