#!/usr/bin/env python
"""
Derived from the "envelope" extension by Aaron Spike, aaron@ekips.org

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
import inkex, os, simplepath, cubicsuperpath
from ffgeom import *
import gettext
_ = gettext.gettext

try:
    from subprocess import Popen, PIPE
    bsubprocess = True
except:
    bsubprocess = False

class Project(inkex.Effect):
    def __init__(self):
            inkex.Effect.__init__(self)

    def effect(self):
        if len(self.options.ids) < 2:
            inkex.errormsg(_("This extension requires two selected paths. \nThe second path must be exactly two nodes long."))
            exit()

        #obj is selected second
        obj = self.selected[self.options.ids[0]]
        trafo = self.selected[self.options.ids[1]]
        if obj.get(inkex.addNS('type','sodipodi')):
            inkex.errormsg(_("The first selected object is of type '%s'.\nTry using the procedure Path->Object to Path." % obj.get(inkex.addNS('type','sodipodi'))))
            exit()
        if obj.tag == inkex.addNS('path','svg') or obj.tag == inkex.addNS('g','svg'):
            if trafo.tag == inkex.addNS('path','svg'):
                #distil trafo into two node points
                trafo = cubicsuperpath.parsePath(trafo.get('d'))
                if len(trafo[0]) < 2:
                    inkex.errormsg(_("This extension requires that the second selected path be two nodes long."))
                    exit()
                trafo = [[Point(csp[1][0],csp[1][1]) for csp in subs] for subs in trafo][0][:2]

                #START HERE!!!
                self.orig = [trafo[0]['x'], trafo[0]['y']]
                self.vect = vect_dif([trafo[1]['x'], trafo[1]['y']], self.orig)
                self.norm = vect_norm(self.vect)
                self.unit = [self.vect[0]/self.norm, self.vect[1]/self.norm]

                if obj.tag == inkex.addNS("path",'svg'):
                    self.process_path(obj)
                if obj.tag == inkex.addNS("g",'svg'):
                    self.process_group(obj)
            else:
                if trafo.tag == inkex.addNS('g','svg'):
                    inkex.errormsg(_("The second selected object is a group, not a path.\nTry using the procedure Object->Ungroup."))
                else:
                    inkex.errormsg(_("The second selected object is not a path.\nTry using the procedure Path->Object to Path."))
                exit()
        else:
            inkex.errormsg(_("The first selected object is not a path.\nTry using the procedure Path->Object to Path."))
            exit()

    def process_group(self,group):
        for node in group:
            if node.tag == inkex.addNS('path','svg'):
                self.process_path(node)
            if node.tag == inkex.addNS('g','svg'):
                self.process_group(node)

    def process_path(self,path):
        d = path.get('d')
        p = cubicsuperpath.parsePath(d)
        for subs in p:
            for csp in subs:
                csp[0] = self.trafopoint(csp[0])
                csp[1] = self.trafopoint(csp[1])
                csp[2] = self.trafopoint(csp[2])
        path.set('d',cubicsuperpath.formatPath(p))

    def trafopoint(self,(x,y)):
        # x1' = -x1 + 2xo + 2n^[(x1 - xo).n^]
        return vect_sub(vect_add(vect_mul(self.orig, 2), vect_mul(self.unit, 2 * dot_prod(vect_sub([x, y], self.orig), self.unit))), [x, y])

def vect_dif((x1, y1), (x2, y2)):
    return [x2-x1, y2-y1]

def sum_sq((x, y)):
    return x**2 + y**2

def vect_norm((x,y)):
    return sum_sq([x, y])**.5

def vect_add((x1, y1), (x2, y2)):
    return [x1+x2, y1+y2]

def vect_sub((x1, y1), (x2, y2)):
    return [x1-x2, y1-y2]

def vect_mul((x, y), s):
    return [x*s, y*s]

def dot_prod((x1, y1), (x2, y2)):
    return (x1 * x2) + (y1 * y2)

if __name__ == '__main__':
    e = Project()
    e.affect()


# vim: expandtab shiftwidth=4 tabstop=8 softtabstop=4 encoding=utf-8 textwidth=99