#!/usr/bin/env python
'''
Copyright (C) 2015 - Ragnar Stiansen

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
'''
import inkex, simplestyle, simplepath, math

class circleBy2Points(inkex.Effect):
    def __init__(self):
        inkex.Effect.__init__(self)
        self.OptionParser.add_option("--tab",
                        action="store", type="string",
                        dest="tab",
                        help="The selected UI-tab when OK was pressed")

        self.OptionParser.add_option( "--deleteLine",
                        action="store", type="inkbool",
                        dest="deleteLine", default=True,
                        help="Deletes the selected line.")

        self.OptionParser.add_option( "--debugMessages",
                        action="store", type="inkbool",
                        dest="debugMessages", default=False,
                        help="Outputs debug messages.")

    def effect(self):
        selection = self.selected
        #Check that one line or path is selected
        if (selection):
            if self.options.debugMessages:
                inkex.errormsg("Number of objects selected: " + str(len(selection)))
            if  len(selection) > 1:
                inkex.errormsg("Please select only one path or line.")
            else:
                # One line or path to process.
                for id, node in selection.iteritems():
                    if node.tag == inkex.addNS('path','svg'):
                        self.addCircle(node)
        else:
            inkex.errormsg("Please select a path or line.")

    def radius( self, x1, y1, x2, y2 ):
         '''
         Pythagorean theorem!
         '''
         if self.options.debugMessages:
            inkex.errormsg('radius called with: ' + str(x1) + ', '\
                            + str(y1) + ', ' + str(x2) + ', ' + str(y2))
         x = x1 - x2
         y = y1 - y2
         return math.sqrt( x * x + y * y )

    def getCoordinates(self, p):
        ''' Extracts the endpoints of line or path.'''
        if self.options.debugMessages:
            inkex.errormsg('getCoordinates called with: \n  ' + str(p))
        #Iterate over path to extract the two endpoint pairs.
        for cmd, params in p:
            if self.options.debugMessages:
                inkex.errormsg('Cmd   : ' + str(cmd))
                inkex.errormsg('Params: ' + str(params))

            circle_att ={}  # Initialize in case next section is never executed

            if cmd == 'M' or cmd == 'm':
                if self.options.debugMessages:
                    inkex.errormsg('Found "M" or "m"')
                # style = { 'stroke':style.c_col, 'stroke-width':str(style.c_th), 'fill':style.c_fill }
                # Extract coordinates of start points
                sx=params[0]
                sy=params[1]
                # Check if both coordinate pairs is in M,
                if len(params) > 2:
                    ex=params[:-2]
                    ey=params[:-1]

            if cmd == 'C' or cmd == 'c' or cmd == 'L' or cmd == 'l':
                # Extract coordinates of end points
                ex=params[-2]
                ey=params[-1]

        # Calculate radius of circle
        r = self.radius(sx, sy, ex, ey)
        if self.options.debugMessages:
            inkex.errormsg('r: '  + str(r) + '\ncx: ' + str(sx) +'\ncy: ' + str(sy))

        circle_att = {
                  'r':  str(r),
                  'cx': str(sx),
                  'cy': str(sy)
                  }
        return circle_att


    def addCircle(self, node):
        # Get group data if any.
        self.group = inkex.etree.SubElement( node.getparent(), inkex.addNS('g','svg') )
        # Get trasforms if any.
        try:
            t = node.get('transform')
            self.group.set('transform', t)
        except:
            pass

        style = simplestyle.parseStyle(node.get('style'))
        a = []
        p = simplepath.parsePath(node.get('d'))
        circle_att = self.getCoordinates(p)
        attribs = {'style':simplestyle.formatStyle(style)}
        # Join the style attributes with CX, CY , R parameters -
        # Normally done with simplestyle and simplepath, could not get it to work ;-)
        z = circle_att.copy()
        z.update(attribs)
        circle_att = z.copy()

        # process the path.
        for cmd,params in p:
            # Find the 'Move' command
            if cmd == 'M' or cmd == 'm':
                inkex.etree.SubElement  (
                                        self.group,
                                        inkex.addNS('circle','svg'),
                                        circle_att
                                        )

        #Delete original line if set in options.
        if self.options.deleteLine:
            node.getparent().remove(node)

if __name__ == '__main__':
    e = circleBy2Points()
    e.affect()
# vim: expandtab shiftwidth=4 tabstop=8 softtabstop=4 fileencoding=utf-8 textwidth=99
