#!/usr/bin/env python
# Inkscape Layer-Based Animation Extensions
# By LordAlexG (lordalexgsoft@gmail.com)
# Export Layers to Sprite Sheet

bEnableDebug = False

# These lines are only needed if you don't put the script directly into
# the installation directory
import sys, os
# Unix
sys.path.append('/usr/share/inkscape/extensions')
# OS X
sys.path.append('/Applications/Inkscape.app/Contents/Resources/extensions')
# Windows
sys.path.append('C:\Program Files\Inkscape\share\extensions')

import inkex
from math import sqrt, ceil

unittouu = inkex.unittouu
uutounit = inkex.uutounit

try:
	from subprocess import Popen, PIPE
	bsubprocess = True
except:
	bsubprocess = False


class ExportSheet(inkex.Effect):
	def __init__(self):
		inkex.Effect.__init__(self)
		self.OptionParser.add_option("--basename",
						action="store", type="string",
						dest="basename", default='export',
						help="")
		self.OptionParser.add_option("--outdir",
						action="store", type="string",
						dest="outdir", default=os.path.abspath('.'),
						help="")
		self.OptionParser.add_option("--inchidden",
						action="store", type="inkbool",
						dest="inchidden", default=False,
						help="")
		self.OptionParser.add_option("--dpi",
						action="store", type="int",
						dest="dpi", default=90,
						help="")

	def effect(self):
		docname			= self.options.basename
		path			= self.options.outdir
		self.bIncHidden	= self.options.inchidden
		self.dpi		= self.options.dpi
		#self.svg_file	= self.args[-1]

		# Find layers
		layers = self.document.xpath("./svg:g[@inkscape:groupmode='layer']", namespaces=inkex.NSS)

		if(bEnableDebug):  self.dbgout = open(os.path.join(path, "debug.txt"), 'w')

		# Locate al layers to be outputted
		layers_list = []
		for l in layers:
			self.search_layer(l, layers_list)
			
		layers_count = len(layers_list)
			
		if(bEnableDebug):  self.dbgout.write(str(layers_count)+" layers found\n")
		
		# Determine sheet dimensions
		sheet_w = int(ceil(sqrt(layers_count)))
		sheet_h = (layers_count+sheet_w-1)/sheet_w
		cell_w = unittouu( self.document.getroot().get('width') )
		cell_h = unittouu( self.document.getroot().get('height') )
		
		# Change page size to sheet size
		self.document.getroot().set('width', str( uutounit(cell_w*sheet_w,'px') ))
		self.document.getroot().set('height', str( uutounit(cell_h*sheet_h,'px') ))
		
		#
		self.sheet_txt = open(os.path.join(path, os.path.join(path, docname)+".txt"), 'w')
		self.cell_w_px = uutounit(cell_w,'px')
		self.cell_h_px = uutounit(cell_h,'px')
		
		# Move layers
		c = 0
		for l in layers_list:
			x = cell_w*(c%sheet_w)
			y = cell_h*(c/sheet_w)
			self.move_layer(l, x, y)
			c += 1
			
		#
		self.sheet_txt.close()
			
		# Write changes to temporal svg
		self.document.write(self.svg_file)
			
		# Export temporal svg to png
		filename = os.path.join(path, docname)+".png"
		command = "inkscape --export-area-page --export-dpi=%d --export-png=\"%s\" \"%s\"" % (self.dpi, filename, self.svg_file)
		
		if(bEnableDebug):  self.dbgout.write(command+'\n')
		
		if bsubprocess:
			p = Popen(command, shell=True, stdout=PIPE, stderr=PIPE)
			return_code = p.wait()
			f = p.stdout
			err = p.stderr
		else:
			_,f,err = os.popen3(command,'r')
		f.close()

		if(bEnableDebug):  self.dbgout.close()
		#end

	def search_layer(self, node, layers_list, name_base=None):
		name = node.get(inkex.addNS('label',u'inkscape'))
		sub = node.xpath("./svg:g[@inkscape:groupmode='layer']", namespaces=inkex.NSS)
		style = node.get('style')
		
		if(bEnableDebug):  self.dbgout.write(str(name)+" -> "+str(style)+"\n")
		
		# Skip hidden if option not set
		if (not style or style=="display:none"):
			if not self.bIncHidden:  return
			else:
				# Show layer
				node.set('style', 'display:inline')
		
		#
		if name_base:
			name_base = name_base+'_'+name
		else:
			name_base = name
		
		# Search sub-layers or add current one
		if sub:
			for l in sub:
				self.search_layer(l, layers_list, name_base)
		else:
			layers_list.append([name_base, node])
			
	def move_layer(self, node_and_name, x, y):
		name = node_and_name[0]
		node = node_and_name[1]
		
		# Add translation to layer
		t = node.get('transform')
		if t:
			t = str(t)+' '
		else:
			t = ''
		
		node.set('transform', t+'translate('+str(x)+','+str(y)+')')
		
		# Write frame data to sheet txt
		x1 = int( uutounit(x,'px')*self.dpi/90 )
		y1 = int( uutounit(y,'px')*self.dpi/90 )
		x2 = int( (x1+self.cell_w_px)*self.dpi/90 )
		y2 = int( (y1+self.cell_h_px)*self.dpi/90 )
		self.sheet_txt.write( "%s\t%d\t%d\t%d\t%d\n" % (name, x1,y1,x2,y2) )
				

if __name__ == '__main__':
	e = ExportSheet()
	e.affect(output=False)
