import xml.etree.ElementTree as ET
import copy
import ol_svg as SG
import os
import subprocess


class format_dict(dict):
    """
    modified dict class
    to prevent key errors for string.format
    """
    def __missing__(self, key):
        return('{'+key+'}')


class OLgenerate():
    """
    generate svg files
    from a svg template
    and data from <ol_data>
    and barcodes from <ol_barcode>
    """

    def __init__(self, filename, printdata):
        self.filename = filename
        self.open(filename)
        self.printdata = printdata
        
        # a dictionary of unit to user unit conversion factors
        self.__uuconv = {'in':90.0, 'pt':1.25, 'px':1.0, 'mm':3.5433070866, 'cm':35.433070866,
                    'm':3543.3070866, 'km':3543307.0866, 'pc':15.0, 'yd':3240.0 , 'ft':1080.0}

        # default options
        self.overwrite = True
        self.naming = '_KEY'
        self.output_path = '.'
        self.make_pdf_flag = False
        self.print_pdf_flag = False
        self.pdf_cmd = '"{svgfile}" --export-pdf="{pdffile}" --export-pdf-version=1.5 --export-dpi=600'
        self.pdf_print = []
        self.prt = None
        self.ink_shell = ['c:/programme/inkscape/inkscape.com', '--shell']

    def open(self, filename):
        self.tmpl = ET.parse(filename)
        self.root = self.tmpl.getroot()
        self.svg_attribs = self.root.attrib


    def save(self, filename):
        outfile = os.path.normpath(os.path.join(self.output_path, filename))
        if self.overwrite != True and os.path.isfile(outfile) == True:
            print(outfile, 'existing and skipped')
            return(False)
        else:
            self.tree.write(outfile)
            return(True)


    def work(self):
        filelist = []
        if self.naming == 'None' or self.naming == None:
            self.naming = '_KEY'
        filter_ids = (('barcode', self.barcode), ('tspan', self.tspan), ('image', self.image), ('flowpara', self.tspan))
        for _key, self.labeldata in enumerate(self.printdata):
            self.labeldata['_KEY'] = _key + 1
            self.tree = copy.deepcopy(self.tmpl)
            for elem in self.tree.iter():
                if 'id' in elem.attrib:
                    id = elem.attrib['id']
                    for filter in filter_ids:
                        if id.lower().startswith(filter[0]):
                            func = filter[1]
                            if func != None:
                                func(elem, format_dict(self.labeldata))
            svgname = str(self.labeldata[self.naming]) + ".svg"
            pdfname = str(self.labeldata[self.naming]) + ".pdf"
            if self.save(svgname):
                filelist.append({'svgfile': os.path.abspath(os.path.normpath(os.path.join(self.output_path, svgname))),
                                 'pdffile': os.path.abspath(os.path.normpath(os.path.join(self.output_path, pdfname))),
                                 'copies': self.labeldata['copies']})

        if self.make_pdf_flag:
            info = subprocess.STARTUPINFO()
            info.dwFlags |= subprocess.STARTF_USESHOWWINDOW
            info.wShowWindow = subprocess.SW_HIDE            
            cmd = subprocess.Popen(self.ink_shell,
                                   stdin=subprocess.PIPE,
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE,
                                   startupinfo=info
                                   )
            
            for entry in filelist:
                shellcmd = self.pdf_cmd.format(**entry)
                shellcmd = shellcmd.replace('\\', '/') # didnt work else on win
                cmd.stdin.write(bytes(shellcmd + '\n', 'utf-8'))
                cmd.stdin.flush()
                print(shellcmd)

            cmd.stdin.write(bytes('quit' + '\n', 'utf-8'))
            cmd.stdin.flush()
            cmd_out, cmd_err = cmd.communicate()
            print(cmd_out.decode(), end="")
            print(cmd_err.decode(), end="")  

        if self.print_pdf_flag:
            for entry in filelist:
                if int(entry['copies']) > 0:
                    cmd = []
                    for para in self.pdf_print:
                        cmd.append(para.format(prt=self.prt,
                                               copies=entry['copies'],
                                               pdf=entry['pdffile']))
                    info = subprocess.STARTUPINFO()
                    info.dwFlags |= subprocess.STARTF_USESHOWWINDOW
                    info.wShowWindow = subprocess.SW_HIDE
                    print(cmd)
                    subprocess.Popen(cmd, startupinfo=info)
                else:
                    print('copies = 0 -> skipped')         


    def pagesize_px(self):
        width = self.svg_attribs['width']
        height = self.svg_attribs['height']
        self.page_width_px = self.unit2px(width)
        self.page_height_px = self.unit2px(height)
        print(self.page_width_px, self.page_height_px)
        

    def unit2px(self, x):
        for i, c in enumerate(''.join(reversed(x))):
            if c.isnumeric():
                unit = (x[i-1:])
                unitvalue = float((x[:i-1]))
                try:
                    unitconv = self.__uuconv[unit]
                except KeyError:
                    unit = 'px'
                    unitconv = self.__uuconv[unit]
                break
        return(unitvalue*unitconv)        


    def image(self, elem, labeldata):
        for attrib in elem.attrib:
            key = attrib.split('}', 1)
            if len(key) == 2:
                key[0] = key[0] + '}'
                if key[1] == 'href':
                    elem.attrib[attrib] = elem.attrib[attrib].format(**labeldata)

        
    def tspan(self, elem, labeldata):
        oldtext = elem.text
        if oldtext != None:
            try:
                elem.text = oldtext.format(**labeldata)
            except KeyError:
                pass
        

    def barcode(self, elem, labeldata):
        tag_only = elem.tag.split('}', 1)[1]
        if tag_only == 'g':
            # remove old barcode
            for subelem in elem.findall(".//"):
                elem.remove(subelem)
            # insert new at same place
            barcode_data = labeldata[elem.attrib['id']]
            barcode_svg = SG.OLsvg()
            barcode_elem = barcode_svg.generateCode(barcode_data)
            for belem in barcode_elem:
                elem.append(belem)
            

def main():
    import ol_data

    labeldata = ol_data.OLData("example.csv")
    xml = OLgenerate('example.svg', labeldata)
    xml.work()
    

if __name__ == '__main__':
    main()    
