#! python
# -*- coding: utf-8 -*-

import sys
import os
import math
import colorsys
import re


_palette_xml_frm = '''<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<svg
  xmlns:dc="http://purl.org/dc/elements/1.1/"
  xmlns:cc="http://creativecommons.org/ns#"
  xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
  xmlns:svg="http://www.w3.org/2000/svg"
  xmlns="http://www.w3.org/2000/svg"
  xmlns:sodipodi="http://sodipodi.sourceforge.net/DTD/sodipodi-0.dtd"
  xmlns:inkscape="http://www.inkscape.org/namespaces/inkscape"
  width="{width}"
  height="{height}"
  viewBox="0 0 {width} {height}"
  id="svg2"
  version="1.1"
  inkscape:version="0.91 r13725"
  sodipodi:docname="{filename}">
  <rect
    id =     "background"
    x =      "0"
    y =      "0"
    width =  "{width}"
    height = "{height}"
    style =  "fill: rgb(0, 0, 0)"/>
{colors}
</svg>
'''
    
_rect_frm = '''  <rect
    id =     "{id}"
    x =      "{x}"
    y =      "{y}"
    width =  "{width}"
    height = "{height}"
    style =  "fill: rgb({r}, {g}, {b})"/>
'''

_palette_regexp = re.compile('^ *(?P<r>[0-9]+) +(?P<g>[0-9]+) +(?P<b>[0-9]+) ?(?P<comment>.*)')

def parse_palette(palette_data):
    global _palette_regexp
    palette = []
    
    for line in palette_data:
        line = line.strip('\n\t\r\b ')
        color_entry = _palette_regexp.match(line)
        if color_entry:
            r = int(color_entry.group('r'))
            g = int(color_entry.group('g'))
            b = int(color_entry.group('b'))
            comment = color_entry.group('comment')
            palette.append( (r, g, b, comment) )
    return palette

def build_palette_svg(filename, palette):
    global _palette_xml_frm
    global _rect_frm
    
    color_width = 32
    color_height = 32
    
    colors = ''
    x = y = 0
    maxX = maxY = 0.0
    index = 0
    while index < len(palette):
        r, g, b, comment = palette[index]
        index += 1
        if 'padding' in comment:
            continue # padding color, skip!
            
        # just for fun swaped x y
        id = 'palette_entry_{i}'.format(i = index)
        xx = y * color_width
        yy = x * color_height
        width = color_width
        height = color_height
        maxX = max(maxX, xx + width)
        maxY = max(maxY, yy + height)
        
        colors += _rect_frm.format(id = id, x = xx, y = yy, width = width, height = height, r = r, g = g, b = b)
        
        if '<br>' in comment:
            y += 1 
            x = 0
        else:
            x += 1
    
    return _palette_xml_frm.format(
            filename = palette_svg,
            width = maxX,
            height = maxY,
            colors = colors)

if __name__ == '__main__':
    # read palette from commandline args
    for filename in sys.argv[1:]:
        palette_data = []
        with open(filename, 'r') as f:
            for line in f:
                palette_data.append(line)

        # parse palette
        palette = parse_palette(palette_data)
        
        # filename without path and extension
        name = os.path.basename(filename).rsplit('.', 1)[0]
        palette_svg = name +'.svg'
        
        svg = build_palette_svg(palette_svg, palette)

        # write svg to file
        with open(palette_svg, 'w') as f:
            f.write(svg)

    
