#!/usr/bin/env python
"""
The MIT License

Copyright (c) 2010 Jairus Martin, frmdstryr@gmail.com

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""
import sys
sys.path.append('/usr/share/inkscape/extensions')
import cubicsuperpath, simplepath, cspsubdiv, simpletransform,bezmisc,copy,math
from simpletransform import *

class HPGL:
	def __init__(self,config):
		self.scale = 1016/90.0
		self.cm = 400.0/self.scale
		self.mm = self.cm/10.0
		self.config = {
			'devSize':(1000,100), # in cm
			'tileSize':(1000,30), # in cm
			'overcut':0.5, # in mm
			'bladeOffset':0.25, # in mm
			'mirror':('false','true'),
			'flatness':0.2, # float
			'boxCopies':False, # t or f
			'position':(0,0), # in cm
			'padding':(5,5), # in mm
			'margin':(10,10), # in mm
			'feed':5 # in mm
		}
		self.config.update(config)
		
	def hpglpreview(self,hpgl,document): # converts the exported hpgl back into svg paths
		svg = document.getroot()
		layer = inkex.etree.SubElement(svg, 'g')
		layer.set(inkex.addNS('label', 'inkscape'), 'InkCut Preview Layer')
		layer.set(inkex.addNS('groupmode', 'inkscape'), 'layer')
		spPU = [['M',[0,0]]]
		spPD = []
		for c in hpgl.split(';'):
			if c[:2] == "PU":
				p = map(int,c[2:].split(','))
				spPD.append(['M',p])
				spPU.append(['L',p])
			elif c[:2] == "PD":
				p = map(int,c[2:].split(','))
				spPU.append(['M',p])
				spPD.append(['L',p])		
		
		pu = inkex.etree.Element(inkex.addNS('path','svg'))
		
		simplepath.scalePath(spPU,0.088582677,-0.088582677)
		
		style = {'fill' : 'none','stroke-opacity': '.5','stroke':'#116AAB'}
		pu.set('style', simplestyle.formatStyle(style))
		pu.set('d',simplepath.formatPath(spPU))
		pu.set('x','0')
		
		pd = inkex.etree.Element(inkex.addNS('path','svg'))
		
		style = {'fill' : 'none','stroke-opacity': '.5','stroke':'#AB3011'}
		pd.set('style', simplestyle.formatStyle(style))
		pd.set('d',simplepath.formatPath(spPD))
		pd.set('x','0')
		# Connect elements together.
		layer.append(pu)
		layer.append(pd)
	
	def svgpreview(self,spl,document,copies=1): # quick way to preview what it will look like... idk if it's useful
		assert type(spl) == list, "export data must be a list of hpgl commands"
		spl = self.adjust(spl,copies)
		svg = document.getroot()
		layer = inkex.etree.SubElement(svg, 'g')
		layer.set(inkex.addNS('label', 'inkscape'), 'InkCut Preview Layer')
		layer.set(inkex.addNS('groupmode', 'inkscape'), 'layer')
		for sp in spl:
			p = inkex.etree.Element(inkex.addNS('path','svg'))
			simplepath.scalePath(sp,0.088582677,-0.088582677)
			simplepath.translatePath(sp,0,self.consumed[1]+self.size[1])
			style = {'fill' : '#20CE39','fill-opacity': '.8','stroke':'#116AAB'}
			p.set('style', simplestyle.formatStyle(style))
			p.set('d',simplepath.formatPath(sp))
			p.set('x','0')
			# Connect elements together.
			layer.append(p)

	
	
	def export(self,spl,copies=1):
		assert type(spl) == list, "export data must be a list of hpgl commands"
		spl = self.adjust(spl,copies)
		if len(spl):
			hpgl = ["IN","SP1","VS4"]
			hpgl.extend(self.convert(spl))
			# back to start	
			hpgl += ["PU%d,0"%((self.consumed[0]+self.size[0]+self.config['feed']*self.mm)*self.scale)]
			
			# print some information
			inkex.debug("Total vinyl used %0.2f (cm)."%(inkex.uutounit(self.consumed[0]+self.size[0],"cm")))
			inkex.debug("Sent %i commands."%(len(hpgl)-3))
			return ";".join(hpgl)+";"
		else:
			return False
		
	def convert(self,spl): 
		# takes in a list of simplepaths [[["M",[x,y]],["L",[x,y]],["L",[x,y]],["Z",[]]],[["M",[x,y]],["L",[x,y]],["L",[x,y]]],...]
		# this can be a list of simplepath.parsePath(node.get("d"))
		# returns list of hpgl commands ["PU0,0","PD100,100",...]
		hpgl = []
		
		# define simplepath comands
		moveto = ['M','m']
		lineto = ['L','l',"H","h","V","v"]
		curveto = ['C','c']
		arc = ["A","a"]
		closepath = ['Z','z']
		# list of simplepaths
		if len(spl):
			for sp in spl:
				if len(sp):
					# decompose simplepath
					first = True
					for c in sp:
						if c[0] in moveto:
							# if first don't append P (doesnt exist yet)
							if first:
								first = False
							else:
								#P.bladeOffset(self.config['bladeOffset']*40)
								P.overcut(self.config['overcut']*40) # 40 mm
								hpgl.extend(P.data)
							
							# make a new path
							P = self.Path()
							P.moveto(c)
						elif c[0] in lineto:
							P.lineto(c)
						elif c[0] in curveto:
							P.curveto(c,self.config['flatness']*self.scale) # what units is flat in?
						# sometimes it doesnt close path when it should... using alternate method
						elif c[0] in closepath: 
							continue
						#	P.overcut(self.config['overcut']*40) # 40 mm
						else:
							# unknown/unchecked command
							#raise ValueError("unknown simplepath command: %s"%(str(c[0])))
							# todo, support a
							inkex.debug("unknown simplepath command: %s"%(str(c[0])))
					# add last paths data
					P.overcut(self.config['overcut']*40) # 40 mm
					#P.bladeOffset(self.config['bladeOffset']*40)
					hpgl.extend(P.data)
				else :
					raise ValueError("simplepath was empty")
		else:
			raise ValueError("no paths to convert")
		return hpgl

		
	
	def adjust(self,spl,clones):
		def bbox(spl): # computes a bbox
			xmin,ymin = spl[0][0][1]
			xmax,ymax = spl[0][0][1]
			for sp in spl:
				csp = cubicsuperpath.CubicSuperPath(sp)
				xmi,xma,ymi,yma = simpletransform.roughBBox(csp)
				xmin = min(xmi,xmin)
				xmax = max(xma,xmax)
				ymin = min(ymi,ymin)
				ymax = max(yma,ymax)
			return xmin,xmax,ymin,ymax
			
		xmin,xmax,ymin,ymax = bbox(spl)
		self.size = (xmax-xmin,ymax-ymin)
		if self.size[0] > (self.config['devSize'][0]*self.cm-self.config['margin'][0]*self.mm) or self.size[1]> (self.config['devSize'][1]*self.cm-self.config['margin'][1]*self.mm):
			raise AssertionError, "path is to large for device, this is not supported yet..."
		
		def offset(spl,d): # adjusts a path for a non-zero blade offset
			# todo ... make this work
			for sp in spl:
				i=0
				for ctl in sp:
					if i>0:
						last = sp[1]
						
					i +=1
			return spl
		
		def translate(spl,dx,dy): # translates a list of paths
			for sp in spl:
				simplepath.translatePath(sp,dx,dy)
			return spl
			
		def scale(spl,sx,sy): # scales a list of paths
			for sp in spl:
				simplepath.scalePath(sp,sx,sy)
			return spl
		
		def copies(spl,c,tile,size,pad): # creates tiles

			d = [0,0]
			clones = []
			first = True
			self.consumed = list(size)
			while c > 0: # make the actual copy commands
				if first:
					first = False
				elif (d[1] + 2*(pad[1] + size[1])) < tile[1]: #add to stack
					d[1] += pad[1] + size[1] # add copy size and padding
				elif (d[0] + pad[0] + size[0]) < tile[0]: # stack full, create row
					d[1] = 0 # reset stack
					d[0] += pad[0] + size[0] # change x to current position plus copy width and padding
					
				self.consumed[1] = max(self.consumed[1],d[1])
				clone = translate(copy.deepcopy(spl),d[0],d[1])
				clones.extend(copy.deepcopy(clone))
				c -=1 # next copy
			
			# this will keep if from starting with the last copy and working backwards
			self.consumed[0] = d[0] # keep track of the maximum x used
			return clones
		
		# move to origin on substrate
		spl = translate(spl,-xmin, -ymin)
		
		# mirror y
		if self.config['mirror'][1] == "true":
			spl = scale(spl,1,-1)
			spl = translate(spl,0,self.size[1])
		
		
		# move to config['position']
		spl = translate(spl,self.config['position'][0]*self.cm, self.config['position'][1]*self.cm+self.config['margin'][1]*self.mm)
		
		maxSize = [self.config['tileSize'][0]*self.cm-self.config['margin'][0]*self.mm,self.config['tileSize'][1]*self.cm-self.config['margin'][1]*self.mm]
		# create copies
		spl = copies(spl,clones,maxSize,self.size,[x * self.mm for x in self.config['padding']])
		
		# scale it
		spl = scale(spl, self.scale, self.scale)
		return spl
	
	def fromSVG(self,nodes):
		# takes in a list of svg node objects
		# returns a list of simplepaths (a sp for each path node)
		spl = []
		for node in nodes:
			tag = node.tag[node.tag.rfind("}")+1:]
			if tag == 'path':
				spl.append(simplepath.parsePath(node.get("d")))
			elif tag == 'rect':
				# how do I convert rect to path??
				raise AssertionError("unknown tag '%s'"%(tag))
			elif tag == 'g':
				spl.extend(self.fromSVG(list(node)))
			else:
				raise AssertionError("unknown tag '%s'"%(tag))
		return spl
	
	class Path:
		def __init__(self):
			self.data = []
			self.sp = []
			
		def moveto(self, sp):
			# svg moveto to hpgl as list, [cmd,[x,y]]
			assert type(sp) == list, 'simplepath must be a list [cmd,[x,y]]'
			self.origin = sp[1]
			self.position = sp[1]
			self.sp.append(sp)
			self.data.append('PU%d,%d'%(sp[1][0],sp[1][1]))
			#return ['PU%d,%d'%(sp[1])]
	
		def lineto(self, sp):
			# svg lineto to hpgl as list, [cmd,[x,y]]
			assert type(sp) == list, 'simplepath must be a list [cmd,[x,y]]'
			self.position = sp[1]
			self.sp.append(sp)
			self.data.append('PD%d,%d'%(sp[1][0],sp[1][1]))
			#return ['PD',sp[1]]
		
		def curveto(self, sp, flat):
			# svg curveto to hpgl as list, [[cmd,[x,y]],[cmd,[x,y]]...]
			assert type(sp) == list, 'simplepath must be a list [cmd,[x,y,x1,y1,x2,y2]]'
			assert sp[0] == 'C', 'simplepath cmd for hpgl.curveto must be: C, given %s' % (sp)
			curve = []
			# build a path for csp parser
			d = simplepath.formatPath([['M',self.position],sp])
			p = cubicsuperpath.parsePath(d)
			cspsubdiv.cspsubdiv(p, flat)
			for sp in p:
				first = True
				for csp in sp:
					#curve.append(['PD',csp[1]])
					if first:
						first = False
					else: 
						self.lineto(csp)
			#return curve

		def overcut(self,d): # used for overcut
			# todo: does not support offset
			dp = bezmisc.pointdistance(self.sp[0][1],self.sp[1][1])
			i = 0
			while d >= 0:
				if d < dp: #last point
					t = d/dp
					self.lineto(['L',list(bezmisc.tpoint(self.sp[i][1],self.sp[i+1][1],t))])
				else: 
					self.lineto(['L',list(self.sp[i+1][1])])
				d -= dp
				i +=1
				dp = bezmisc.pointdistance(self.sp[i][1],self.sp[i+1][1])
				
		def bladeOffset(self,d): # adjusts a path for a blade offset, uses a complete path
		# todo, this has me beat...
			def dotP(p0,p1):
				p = 0
				for a1,a2 in zip(p0,p1):
					p +=a1*a2
				return p
			def norm(p0):
				n = 0
				for a in p0:
					n +=a*a
				return math.sqrt(n)	
				
			
			if d<=0 or len(self.sp) < 3:
				return None
			i = 1
			angle = 0.1
			while i < (len(self.sp)-1):
				next = self.sp[i+1]
				cur = self.sp[i]
				last = self.sp[i-1]
				# get the angle between the current and last line
				p0 = [float(cur[1][0]-last[1][0]),float(cur[1][1]-last[1][1])]
				p1 = [float(next[1][0]-cur[1][0]),float(next[1][1]-cur[1][1])]
				theta = math.acos(dotP(p0,p1)/(norm(p0)*norm(p1)))
				inkex.debug(theta)
				
				dp = bezmisc.pointdistance(next[1],cur[1])
				t = 1-d/dp
				self.data.insert(i+1,'PD%d,%d'%(bezmisc.tpoint(next[1],cur[1],t)))
				# extend the path the offset distance
				dp = bezmisc.pointdistance(last[1],cur[1])
				t = d/dp+1
				self.data[i] = 'PD%d,%d'%(bezmisc.tpoint(last[1],cur[1],t))
				# backtrack to the original line
					
				i+=1
				
				
			

