#!/usr/bin/env python
"""
The MIT License

Copyright (c) 2010 Jairus Martin, frmdstryr@gmail.com

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""
# These two lines are only needed if you don't put the script directly into
# the installation directory
import sys
sys.path.append('/usr/share/inkscape/extensions')

# We will use the inkex module with the predefined Effect base class.
import inkex, hpgl
# The simplestyle module provides functions for style parsing.

class inkcut(inkex.Effect):
	def __init__(self):
		# Call the base class constructor.
		inkex.Effect.__init__(self)
		# Define output options
		self.OptionParser.add_option('--tab', action = 'store',
		  type = 'string', dest = 'tab')
		
		self.OptionParser.add_option('-f', '--flat', action = 'store',
		  type = 'float', dest = 'flat', default = 0.2,
		  help = 'Number of copies you want to cut.')
		  
		self.OptionParser.add_option('-c', '--copies', action = 'store',
		  type = 'int', dest = 'copies', default = 1,
		  help = 'Number of copies you want to cut.')
		self.OptionParser.add_option('--box', action = 'store',
		   dest = 'box', default = False,
		  help = 'Cut a box around copies.')
		
		self.OptionParser.add_option('--hpad', action = 'store',
		  type = 'float', dest = 'padx', default = .1,
		  help = 'Padding inbetween horizontal copies in a stack.')
       
		self.OptionParser.add_option('--vpad', action = 'store',
		  type = 'float', dest = 'pady', default = .1,
		  help = 'Padding between vertical stacks of copies.')
		
		self.OptionParser.add_option('--origin', action = 'store',
		  dest = 'origin', default = True,
		  help = 'Auto detect origin.')
		self.OptionParser.add_option('-x', '--xorg', action = 'store',
		  type = 'float', dest = 'x0', default = 0,
		  help = 'Origin x position.')
       
		self.OptionParser.add_option('-y', '--yorg', action = 'store',
		  type = 'float', dest = 'y0', default = 0,
		  help = 'Origin y position.')       
		self.OptionParser.add_option('--xoff', action = 'store',
		  type = 'float', dest = 'xoff', default = 0,
		  help = 'Offset x position.')
       
		self.OptionParser.add_option('--yoff', action = 'store',
		  type = 'float', dest = 'yoff', default = 0,
		  help = 'Offset y position.')       
		self.OptionParser.add_option('--feed', action = 'store',
		  type = 'float', dest = 'feed', default = True,
		  help = 'Distance to feed at the end of plot.')       
		# Define device options
		self.OptionParser.add_option('-r', '--rotation', action = 'store',
		  dest = 'swap', default = True,
		  help = 'Rotate plotting axis 90 degrees.')
		
		self.OptionParser.add_option('--mirrory', action = 'store',
		  dest = 'my', default = True,
		  help = 'Mirror x-axis.')
		
		self.OptionParser.add_option('--overcut', action = 'store',
		  type='float', dest = 'overcut', default = .02,
		  help = 'Amount to overcut paths.')
		self.OptionParser.add_option('--bladeoffset', action = 'store',
		  type='float', dest = 'bladeoffset', default = .025,
		  help = 'Offset of blade.')
		
		self.OptionParser.add_option('--deviceWidth', action = 'store',
		  type = 'int', dest = 'deviceWidth', default = 10,
		  help = 'Maximum device plotting/cutting width in cm.')
		
		self.OptionParser.add_option('--deviceLength', action = 'store',
		  type = 'int', dest = 'deviceLength', default = 0,
		  help = 'Maximum device plotting/cutting width in cm.')
		
		self.OptionParser.add_option('--tileWidth', action = 'store',
		  type = 'int', dest = 'tileWidth', default = 10,
		  help = 'Maximum tile/vinyl substrate width in cm.')
		
		self.OptionParser.add_option('--tileLength', action = 'store',
		  type = 'int', dest = 'tileLength', default = 1000,
		  help = 'Maximum device plotting/cutting width in cm.')
		  
		self.OptionParser.add_option('--margin', action = 'store',
		  type = 'int', dest = 'margin', default = 1,
		  help = 'Tile margins in cm.')
		self.OptionParser.add_option('-d','--path', action = 'store',
		  type = 'string', dest = 'path', default = 0,
		  help = 'ID of the path you want to cut.')
		
		self.OptionParser.add_option('--preview', action = 'store',
		  dest = 'preview', default = False,
		  help = 'Preview instead of cutting')
		
		# Define serial port settings
		self.OptionParser.add_option('-p','--port', action = 'store',
		  type = 'string', dest = 'port', default = '/dev/ttyUSB0',
		  help = 'Serial port index you want to use.')
		
		self.OptionParser.add_option('-b','--baud', action = 'store',
		  type = 'int', dest = 'baud', default = 9600,
		  help = 'Serial port baud rate you want to use.')
		
		self.OptionParser.add_option('--bytesize', action = 'store',
		  type = 'int', dest = 'bytesize', default = 8,
		  help = 'Serial port bytesize you want to use.')
		
		self.OptionParser.add_option('--parity', action = 'store',
		  type = 'string', dest = 'parity', default = "N",
		  help = 'Serial port parity you want to use.')
		
		self.OptionParser.add_option('--stopbits', action = 'store',
		  type = 'int', dest = 'stopbits', default = 1,
		  help = 'Serial port stopbits you want to use.')
		  
		self.OptionParser.add_option('--xonxoff', action = 'store',
		  dest = 'xonxoff', default = False,
		  help = 'Enable serial port xonxoff.')
		
		self.OptionParser.add_option('--rtscts', action = 'store',
		  dest = 'rtscts', default = True,
		  help = 'Enable serial port rtscts.')
		

		
    
	
	
	
	def plot(self,data,settings):
		assert type(data) == str, "input data must be a str type"
		assert type(settings) == dict, "port settings must be a dict type"
		import serial

		# set default settings
		set = {'baud':9600,'port':2}
		set.update(settings);
		
		#create serial and set settings
		ser = serial.Serial()
		ser.baudrate = set['baud']
		ser.port = set['port']
		ser.open()
		if ser.isOpen():
			#send data & return bits sent
			bits = ser.write(data);
			ser.close();
			if bits == None:
				#inkex.errormsg("Error sending data to plotter, no bits were sent")
				#inkex.debug(ser)
				return False
			else:
				inkex.errormsg("inkcut.plot() sucessfully sent %d bits" % (bits))
				return bits;
			
		else:
			#inkex.errormsg("Error sending to plotter, port %s could not be opened" % set['port'])
			#inkex.debug(ser)
			return False;


	
	
	
	def copies(self,copies,nodes,xsize,ysize,xpos=0,ypos=0,xmax=160,ymax=1000,xpad=.2,ypad=.2,xmargin=1,ymargin=1):
		""" 
		used for tiling copies of a path
		returns an array with all the hpgl commands in a list
		"""
		#check inputs
		assert type(nodes) == list, "nodes must be a list"
		assert copies > 0, "must have at least 1 copy"
		assert xmax > xsize, "max x (%d) must be more than path x size (%d)" % (xmax,xsize)
		assert ymax > ysize, "max y (%d) must be more than path y size (%d)" % (ymax,ysize)
		assert xpos < xmax, "initial x pos (%d) must be less than max (%d)" % (xpos,xmax)
		assert ypos < ymax, "initial y pos (%d) must be less than max (%d)" % (ypos,ymax)
		
		#check if rotating will save space
		def autorotate(max,xsize,ysize,xpad,xmargin):
			# check if there is more un used space, if so rotate
			if (xmax-(2*xmargin))%(xpad+xsize) <= (xmax-(2*xmargin))%(xpad+ysize):
				rotate = False
			else :
				rotate = True
			return rotate

		data = []
		first=True
		while copies > 0:
			if first:
				x0,y0 = xpos,ypos
				first = False
			elif (xpos + xpad + xsize) < (xmax - (2*xmargin) ): #add to stack
				xpos += xpad + xsize # add copy size and padding
			elif (ypos + ypad + ysize) < (ymax - (2*ymargin) ): # stack full, create row
				xpos = x0 # reset stack
				ypos = ypos + ypad + ysize # change y to current position plus copy width and padding
			
			data += self.hpgl(nodes,xpos,ypos)
			copies -=1 # next copy
		
		return data



	def effect(self):
		# rename options.
		options = self.options
		selected = self.selected
		# Get access to main SVG document element and get its dimensions.
		svg = self.document.getroot()
		nodes = selected.values()
		assert len(nodes) > 0, inkex.errormsg("no paths were selected to plot")
		
		# set HPGL config
		config = {
			'devSize':(self.options.deviceLength,self.options.deviceWidth),
			'tileSize':(self.options.tileLength,self.options.tileWidth),
			'overcut':self.options.overcut,
			'bladeOffset':self.options.bladeoffset,
			'mirror':(1,self.options.my),
			'flatness':self.options.flat, # float
			'boxCopies':self.options.box, # t or f
			'position':(self.options.xoff,self.options.yoff),
			'padding':(self.options.padx,self.options.pady),
			'margin':(self.options.margin,self.options.margin),
			'feed':self.options.feed
		}
		
		HPGL = hpgl.HPGL(config)
		# get paths from SVG
		spl = HPGL.fromSVG(nodes)
		
		if self.options.preview == "false":
			plotsettings = {'port':options.port,
				'baud':options.baud,
				'bytesize':options.bytesize,
				'stopbits':options.stopbits,
				'parity':options.parity,
				'xonxoff':options.xonxoff,
				'rtscts':options.rtscts}
			# convert to hpgl
			plotdata = HPGL.export(spl,self.options.copies)
			inkex.debug(plotdata)
			# send it to cutter
			self.plot(plotdata,plotsettings)
		else: 
			HPGL.hpglpreview(HPGL.export(spl,self.options.copies),self.document)
		
		

# Create effect instance and apply it.
effect = inkcut()
effect.affect()
